package com.smart.hospital.common.pay.factory;

import com.smart.hospital.common.pay.source.PayAttribute;
import com.smart.hospital.common.pay.source.PayAttributeSource;
import com.smart.hospital.common.pay.strategy.PayStrategy;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.UsesJava7;

import java.io.Serializable;
import java.lang.invoke.MethodHandles;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

@Slf4j
public class PayClientProxy<T> implements InvocationHandler, Serializable {

	private static final long serialVersionUID = 1L;

	/**
	 * 代理的接口类
	 */
	private final Class<T> clientInterface;

	/**
	 * 注解解析器
	 */
	private final PayAttributeSource payAttributeSource;

	/**
	 * 支付策略
	 */
	private final PayStrategy payStrategy;

	public PayClientProxy(Class<T> clientInterface, PayAttributeSource payAttributeSource, PayStrategy payStrategy) {
		this.clientInterface = clientInterface;
		this.payAttributeSource = payAttributeSource;
		this.payStrategy = payStrategy;
	}

	@Override
	public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
		if (Object.class.equals(method.getDeclaringClass())) {
			return method.invoke(this, args);
		} else if (isDefaultMethod(method)) {
			return invokeDefaultMethod(proxy, method, args);
		}
		// 解析注解信息
		PayAttribute attr = parseAnnotationMetaInfo(method);
		// 调用支付
		return getPayStrategy().doExecute(attr, clientInterface, method, args);
	}

	@UsesJava7
	private Object invokeDefaultMethod(Object proxy, Method method, Object[] args)
			throws Throwable {
		final Constructor<MethodHandles.Lookup> constructor = MethodHandles.Lookup.class
				.getDeclaredConstructor(Class.class, int.class);
		if (!constructor.isAccessible()) {
			constructor.setAccessible(true);
		}
		final Class<?> declaringClass = method.getDeclaringClass();
		return constructor
				.newInstance(declaringClass,
						MethodHandles.Lookup.PRIVATE | MethodHandles.Lookup.PROTECTED
								| MethodHandles.Lookup.PACKAGE | MethodHandles.Lookup.PUBLIC)
				.unreflectSpecial(method, declaringClass).bindTo(proxy).invokeWithArguments(args);
	}

	/**
	 * Backport of java.lang.reflect.Method#isDefault()
	 */
	private boolean isDefaultMethod(Method method) {
		return (method.getModifiers()
				& (Modifier.ABSTRACT | Modifier.PUBLIC | Modifier.STATIC)) == Modifier.PUBLIC
				&& method.getDeclaringClass().isInterface();
	}

	private PayAttribute parseAnnotationMetaInfo(Method method) {
		return getPayAttributeSource().getPayAttribute(method, clientInterface);
	}

	public PayAttributeSource getPayAttributeSource() {
		return payAttributeSource;
	}

	public PayStrategy getPayStrategy() {
		return payStrategy;
	}
}
