package com.smart.hospital.common.pay.config;

import com.smart.hospital.common.pay.factory.PayFactoryBean;
import com.smart.hospital.common.pay.source.PayAttributeSource;
import com.smart.hospital.common.pay.strategy.PayStrategy;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.AnnotatedBeanDefinition;
import org.springframework.beans.factory.config.BeanDefinitionHolder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.context.annotation.ClassPathBeanDefinitionScanner;

import java.util.Arrays;
import java.util.Set;

@Slf4j
public abstract class AbstractClientRegistryScanner extends ClassPathBeanDefinitionScanner {

	private PayAttributeSource payAttributeSource;

	private PayStrategy payStrategy;

	protected PayFactoryBean<?> factoryBean = new PayFactoryBean<>();

	public AbstractClientRegistryScanner(BeanDefinitionRegistry registry) {
		super(registry, false);
	}

	public PayFactoryBean<?> getFactoryBean() {
		return factoryBean;
	}

	public PayAttributeSource getPayAttributeSource() {
		return payAttributeSource;
	}

	public void setPayAttributeSource(PayAttributeSource payAttributeSource) {
		this.payAttributeSource = payAttributeSource;
	}

	public void setPayStrategy(PayStrategy payStrategy) {
		this.payStrategy = payStrategy;
	}

	public PayStrategy getPayStrategy() {
		return payStrategy;
	}

	@Override
	protected Set<BeanDefinitionHolder> doScan(String... basePackages) {
		Set<BeanDefinitionHolder> definitions = super.doScan(basePackages);
		if (definitions.isEmpty()) {
			log.warn("No interface was found in '" + Arrays.toString(basePackages) + "' package. Please check your configuration.");
		} else {
			processBeanDefinitions(definitions);
		}
		return definitions;
	}

	@Override
	protected boolean isCandidateComponent(AnnotatedBeanDefinition beanDefinition) {
		return beanDefinition.getMetadata().isInterface() && beanDefinition.getMetadata().isIndependent();
	}

	public void registerFilters() {

		boolean acceptAllInterfaces = true;

		if (acceptAllInterfaces) {
			// default include filter that accepts all classes
			addIncludeFilter((metadataReader, metadataReaderFactory) -> true);
		}

		// exclude package-info.java
		addExcludeFilter((metadataReader, metadataReaderFactory) -> {
			String className = metadataReader.getClassMetadata().getClassName();
			return className.endsWith("package-info");
		});
	}

	/**
	 * 处理Bean
	 *
	 * @param beanDefinitions
	 */
	protected abstract void processBeanDefinitions(Set<BeanDefinitionHolder> beanDefinitions);
}
