package com.smart.hospital.common.lock.provider.pattern;

import cn.hutool.core.util.StrUtil;
import com.smart.hospital.common.lock.config.InitConfig;
import com.smart.hospital.common.lock.config.RedisSchemeConfig;
import com.smart.hospital.common.lock.config.redis.RedisClusterConfig;
import com.smart.hospital.common.lock.constant.LoadBalancerTypeConstant;
import com.smart.hospital.common.lock.constant.LockCommonConstant;
import com.smart.hospital.common.lock.constant.SubReadModeTypeConstant;
import com.smart.hospital.common.lock.enums.ServerPattern;
import org.redisson.Redisson;
import org.redisson.api.RedissonClient;
import org.redisson.config.ClusterServersConfig;
import org.redisson.config.ReadMode;
import org.redisson.config.SubscriptionMode;
import org.redisson.connection.balancer.LoadBalancer;
import org.redisson.connection.balancer.RandomLoadBalancer;
import org.redisson.connection.balancer.RoundRobinLoadBalancer;
import org.redisson.connection.balancer.WeightedRoundRobinBalancer;
import org.springframework.util.Assert;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

public class RedisClusterProvider implements PatternProvider {

	@Override
	public RedissonClient init(InitConfig baseConfig) {
		ClusterServersConfig clusterServerConfig = baseConfig.useClusterServers();
		RedisSchemeConfig redisSchemeConfig = baseConfig.getLockProperties().getRedis();
		RedisClusterConfig clusterConfig = redisSchemeConfig.getClusterServer();

		String[] addressArr = clusterConfig.getNodeAddresses().split(LockCommonConstant.COMMA);
		Arrays.asList(addressArr).forEach(address -> clusterServerConfig
				.addNodeAddress(String.format("%s%s", LockCommonConstant.REDIS_URL_PREFIX, address)));
		clusterServerConfig.setScanInterval(clusterConfig.getScanInterval());

		ReadMode readMode = getReadMode(clusterConfig.getReadMode());
		Assert.notNull(readMode, "未知读取操作的负载均衡模式类型");
		clusterServerConfig.setReadMode(readMode);

		SubscriptionMode subscriptionMode = getSubscriptionMode(clusterConfig.getSubMode());
		Assert.notNull(readMode, "未知订阅操作的负载均衡模式类型");
		clusterServerConfig.setSubscriptionMode(subscriptionMode);

		LoadBalancer loadBalancer = getLoadBalancer(clusterConfig.getLoadBalancer(), clusterConfig.getWeightMaps(),
				clusterConfig.getDefaultWeight());
		Assert.notNull(readMode, "未知负载均衡算法类型");
		clusterServerConfig.setLoadBalancer(loadBalancer);

		clusterServerConfig.setSubscriptionConnectionMinimumIdleSize(clusterConfig.getSubConnMinIdleSize());
		clusterServerConfig.setSubscriptionConnectionPoolSize(clusterConfig.getSubConnPoolSize());
		clusterServerConfig.setSlaveConnectionMinimumIdleSize(clusterConfig.getSlaveConnMinIdleSize());
		clusterServerConfig.setSlaveConnectionPoolSize(clusterConfig.getSlaveConnPoolSize());
		clusterServerConfig.setMasterConnectionMinimumIdleSize(clusterConfig.getMasterConnMinIdleSize());
		clusterServerConfig.setMasterConnectionPoolSize(clusterConfig.getMasterConnPoolSize());
		clusterServerConfig.setIdleConnectionTimeout(clusterConfig.getIdleConnTimeout());
		clusterServerConfig.setConnectTimeout(clusterConfig.getConnTimeout());
		clusterServerConfig.setTimeout(clusterConfig.getTimeout());
		clusterServerConfig.setRetryAttempts(clusterConfig.getRetryAttempts());
		clusterServerConfig.setRetryInterval(clusterConfig.getRetryInterval());
		clusterServerConfig.setPassword(clusterConfig.getPassword());
		clusterServerConfig.setSubscriptionsPerConnection(clusterConfig.getSubPerConn());
		clusterServerConfig.setClientName(redisSchemeConfig.getClientName());
		return Redisson.create(baseConfig);
	}

	@Override
	public boolean supports(String pattern) {
		if (StrUtil.isBlank(pattern)) {
			return false;
		}
		return ServerPattern.CLUSTER.getPattern().equals(pattern);
	}

	/**
	 * 根据readModeType返回ReadMode
	 *
	 * @param readModeType 读取操作的负载均衡模式类型
	 * @return ReadMode OR NULL
	 */
	private ReadMode getReadMode(String readModeType) {
		if (SubReadModeTypeConstant.SLAVE.equals(readModeType)) {
			return ReadMode.SLAVE;
		}
		if (SubReadModeTypeConstant.MASTER.equals(readModeType)) {
			return ReadMode.MASTER;
		}
		if (SubReadModeTypeConstant.MASTER_SLAVE.equals(readModeType)) {
			return ReadMode.MASTER_SLAVE;
		}
		return null;
	}

	/**
	 * 根据subscriptionModeType返回SubscriptionMode
	 *
	 * @param subscriptionModeType 订阅操作的负载均衡模式类型
	 * @return SubscriptionMode OR NULL
	 */
	private SubscriptionMode getSubscriptionMode(String subscriptionModeType) {
		if (SubReadModeTypeConstant.SLAVE.equals(subscriptionModeType)) {
			return SubscriptionMode.SLAVE;
		}
		if (SubReadModeTypeConstant.MASTER.equals(subscriptionModeType)) {
			return SubscriptionMode.MASTER;
		}
		return null;
	}

	/**
	 * 根据用户的配置类型设置对应的LoadBalancer
	 *
	 * @param loadBalancerType   负载均衡算法类名
	 * @param customerWeightMaps 权重值设置，当负载均衡算法是权重轮询调度算法时该属性有效
	 * @param defaultWeight      默认权重值，当负载均衡算法是权重轮询调度算法时该属性有效
	 * @return LoadBalancer OR NULL
	 */
	private LoadBalancer getLoadBalancer(String loadBalancerType, String customerWeightMaps, int defaultWeight) {
		if (LoadBalancerTypeConstant.RANDOM_LOAD_BALANCER.equals(loadBalancerType)) {
			return new RandomLoadBalancer();
		}
		if (LoadBalancerTypeConstant.ROUND_ROBIN_LOAD_BALANCER.equals(loadBalancerType)) {
			return new RoundRobinLoadBalancer();
		}
		if (LoadBalancerTypeConstant.WEIGHTED_ROUND_ROBIN_BALANCER.equals(loadBalancerType)) {
			Map<String, Integer> weights = new HashMap<>(16);
			String[] weightMaps = customerWeightMaps.split(LockCommonConstant.SEMICOLON);
			Arrays.asList(weightMaps)
					.forEach(weightMap -> weights.put(
							LockCommonConstant.REDIS_URL_PREFIX + weightMap.split(LockCommonConstant.COMMA)[0],
							Integer.parseInt(weightMap.split(LockCommonConstant.COMMA)[1])));
			return new WeightedRoundRobinBalancer(weights, defaultWeight);
		}
		return null;
	}

}
