package com.smart.hospital.common.lock.advisor.source;

import com.smart.hospital.common.lock.advisor.attribute.LockAttribute;
import com.smart.hospital.common.lock.key.LockKey;
import org.springframework.aop.support.AopUtils;
import org.springframework.core.MethodClassKey;
import org.springframework.lang.Nullable;
import org.springframework.util.ClassUtils;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public abstract class AbstractDistributedLockAttributeSource implements LockAttributeSource {

	/**
	 * @Lock注解属性的缓存
	 */
	private final Map<Object, LockAttribute> attributeCache = new ConcurrentHashMap<>(16);

	/**
	 * 锁的操作对象缓存
	 */
	private final Map<Object, LockKey<String, List<String>>> lockServiceCache = new ConcurrentHashMap<>(16);

	@Override
	public LockAttribute getLockAttribute(Method method, Class<?> targetClass) {
		if (method.getDeclaringClass() == Object.class) {
			return null;
		}

		// 先从本地缓存中取
		Object cacheKey = getCacheKey(method, targetClass);

		LockAttribute cached = this.attributeCache.get(cacheKey);

		if (cached != null) {
			return cached;
		}

		// 解析注解信息
		LockAttribute attr = computeLockAttribute(method, targetClass);

		if (attr != null) {
			//  将信息放入本地缓存
			this.attributeCache.put(cacheKey, attr);
		}

		return attr;
	}

	@Override
	public LockKey<String, List<String>> getLockKey(Method method, Class<?> targetClass, Object[] arguments, LockAttribute attr) {
		if (method.getDeclaringClass() == Object.class) {
			return null;
		}

		Object cacheKey = getCacheKey(method, targetClass);

		LockKey<String, List<String>> cached = this.lockServiceCache.get(cacheKey);
		if (cached != null) {
			return cached;
		}

		Method specificMethod = AopUtils.getMostSpecificMethod(method, targetClass);

		LockKey<String, List<String>> lockKey = findLockKey(specificMethod, targetClass, arguments, attr);
		if (lockKey != null) {
			return lockKey;
		}
		return null;
	}

	protected LockAttribute computeLockAttribute(Method method, Class<?> targetClass) {
		// 只允许public公共方法 并且 方法不是公共方法 = 不是公共方法匹配失败
		if (allowPublicMethodOnly() && !Modifier.isPublic(method.getModifiers())) {
			return null;
		}

		// 从代理对象上的一个方法，找到真实对象上对应的方法。
		// 举个例子，MyComponent代理之后的对象上的someLogic方法，肯定是属于cglib代理之后的类上的method，
		// 使用这个method是没法去执行目标MyComponent的someLogic方法
		Method specificMethod = AopUtils.getMostSpecificMethod(method, targetClass);

		// First try is the method in the target class
		LockAttribute attr = findLockAttribute(specificMethod);
		if (attr != null) {
			return attr;
		}

		// Second try is the lock attribute on the target class.
		attr = findLockAttribute(targetClass);
		if (attr != null && ClassUtils.isUserLevelMethod(method)) {
			return attr;
		}

		if (specificMethod != method) {
			attr = findLockAttribute(method);
			if (attr != null) {
				return attr;
			}
			attr = findLockAttribute(method.getDeclaringClass());
			if (attr != null && ClassUtils.isUserLevelMethod(method)) {
				return attr;
			}
		}
		return null;
	}

	/**
	 * 从方法解析@Lock注解信息
	 *
	 * @param method
	 * @return
	 */
	protected abstract LockAttribute findLockAttribute(Method method);

	/**
	 * 从类解析@Lock注解信息
	 *
	 * @param clazz
	 * @return
	 */
	protected abstract LockAttribute findLockAttribute(Class<?> clazz);

	/**
	 * 从方法上获取注解信息
	 *
	 * @param method
	 * @return
	 */
	protected abstract LockKey<String, List<String>> findLockKey(Method method, Class<?> targetClass, Object[] arguments, LockAttribute attr);

	/**
	 * 只匹配public的方法创建代理
	 *
	 * @return
	 */
	protected boolean allowPublicMethodOnly() {
		return false;
	}

	protected Object getCacheKey(Method method, @Nullable Class<?> targetClass) {
		return new MethodClassKey(method, targetClass);
	}

}
