package com.smart.hospital.common.lock.advisor.parser;

import com.smart.hospital.common.lock.advisor.attribute.LockAttribute;
import com.smart.hospital.common.lock.annotation.Lock;
import com.smart.hospital.common.lock.annotation.RLock;
import com.smart.hospital.common.lock.annotation.ZLock;
import com.smart.hospital.common.lock.enums.LockScheme;
import com.smart.hospital.common.lock.key.strategy.*;
import com.smart.hospital.common.lock.key.strategy.impl.ClassKeyStrategy;
import com.smart.hospital.common.lock.key.strategy.impl.FieldKeyStrategy;
import com.smart.hospital.common.lock.key.strategy.impl.MethodKeyStrategy;
import com.smart.hospital.common.lock.key.strategy.impl.ParameterKeyStrategy;
import org.springframework.core.annotation.AnnotatedElementUtils;
import org.springframework.core.annotation.AnnotationAttributes;

import java.lang.annotation.Annotation;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;

public abstract class AbstractLockAnnotationParser implements LockAnnotationParser {

	/**
	 * 所有的所注解类型
	 */
	private static final Class<? extends Annotation>[] LOCK_ANNOTATION_CLASSES = new Class[]{Lock.class, RLock.class, ZLock.class};

	/**
	 * 所有作为key的标记的注解 目前用@Key
	 */
	private static final Class<? extends Annotation> KEY_ANNOTATION_CLASS = KeyStrategy.KEY_ANNOTATION_CLASS;

	@Override
	public LockAttribute parseLockAnnotation(AnnotatedElement element) {
		// 找到注解信息
		for (Class<? extends Annotation> clazz : LOCK_ANNOTATION_CLASSES) {
			AnnotationAttributes attributes = AnnotatedElementUtils.findMergedAnnotationAttributes(element, clazz, false, false);
			if (attributes != null) {
				return parseLockAnnotation(attributes);
			}
		}
		return null;
	}

	@Override
	public KeyStrategy parseKeyAnnotation(Method method, Class<?> targetClass, Object[] arguments, LockAttribute attr) {
		return getKeyStrategyFromMethod(method, targetClass, arguments, attr);
	}

	@Override
	public LockScheme getLockScheme(Method realMethod) {
		if (null != realMethod.getAnnotation(Lock.class) || null != realMethod.getAnnotation(RLock.class)) {
			return LockScheme.REDIS;
		}
		return LockScheme.ZOOKEEPER;
	}

	private KeyStrategy getKeyStrategyFromMethod(Method method, Class<?> targetClass, Object[] arguments, LockAttribute attr) {
		LockScheme scheme = getLockScheme(method);
		// 参数锁
		Parameter[] parameters = method.getParameters();
		if (parameters != null && parameters.length > 0) {
			for (int i = 0; i < parameters.length; i++) {
				if (parameters[i].isAnnotationPresent(KEY_ANNOTATION_CLASS)) {
					return new ParameterKeyStrategy(targetClass.getName(), method.getName(), method, arguments, scheme, attr);
				}
			}
		}
		// 方法锁
		if (null != method.getAnnotation(KEY_ANNOTATION_CLASS)) {
			return new MethodKeyStrategy(targetClass.getName(), method.getName(), method, arguments, scheme, attr);
		}
		// 属性锁
		if (arguments != null && arguments.length > 0) {
			for (int i = 0; i < arguments.length; i++) {
				Object argument = arguments[i];
				if (argument == null) {
					continue;
				}
				Class<?> argumentClass = argument.getClass();
				Field[] fields = argumentClass.getDeclaredFields();
				for (Field field : fields) {
					if (null != field.getAnnotation(KEY_ANNOTATION_CLASS)) {
						return new FieldKeyStrategy(targetClass.getName(), method.getName(), method, arguments, scheme, attr);
					}
				}
			}
		}
		// 类名和方法名作为key的锁
		return new ClassKeyStrategy(targetClass.getName(), method.getName(), method, arguments, scheme, attr);
	}

	protected abstract LockAttribute parseLockAnnotation(AnnotationAttributes attributes);

	protected abstract LockAttribute parseLockAnnotation(Annotation annotation);
}
