package com.smart.hospital.common.lock;

import com.smart.hospital.common.lock.advisor.LockInterceptorAutoConfiguration;
import com.smart.hospital.common.lock.config.InitConfig;
import com.smart.hospital.common.lock.config.ZkRetryPolicyStore;
import com.smart.hospital.common.lock.config.ZookeeperSchemeConfig;
import com.smart.hospital.common.lock.config.zookeeper.ExponentialBackoffRetryConfig;
import com.smart.hospital.common.lock.config.zookeeper.ForeverRetryConfig;
import com.smart.hospital.common.lock.config.zookeeper.NTimesRetryConfig;
import com.smart.hospital.common.lock.config.zookeeper.UntilElapsedRetryConfig;
import com.smart.hospital.common.lock.constant.LockCommonConstant;
import com.smart.hospital.common.lock.custom.PropertiesCustomizer;
import com.smart.hospital.common.lock.custom.impl.CodecPropertiesCustomizer;
import com.smart.hospital.common.lock.enums.LockScheme;
import com.smart.hospital.common.lock.enums.ZkRetryPolicy;
import com.smart.hospital.common.lock.properties.LockProperties;
import com.smart.hospital.common.lock.provider.manager.PatternManager;
import com.smart.hospital.common.lock.provider.manager.RedisServerPatternManager;
import com.smart.hospital.common.lock.provider.pattern.PatternProvider;
import com.smart.hospital.common.lock.provider.pattern.RedisClusterProvider;
import com.smart.hospital.common.lock.provider.pattern.RedisSingleProvider;
import com.smart.hospital.common.lock.service.LockProviderManager;
import com.smart.hospital.common.lock.service.LockServiceManager;
import com.smart.hospital.common.lock.service.provider.LockServiceProvider;
import com.smart.hospital.common.lock.service.provider.redis.RedisFairLockServiceProvider;
import com.smart.hospital.common.lock.service.provider.redis.RedisMultiLockServiceProvider;
import com.smart.hospital.common.lock.service.provider.redis.RedisReentrantLockServiceProvider;
import com.smart.hospital.common.lock.service.provider.zookeeper.ZookeeperMultiZkLockServiceProvider;
import lombok.SneakyThrows;
import org.apache.curator.RetryPolicy;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.ExponentialBackoffRetry;
import org.apache.curator.retry.RetryForever;
import org.apache.curator.retry.RetryNTimes;
import org.apache.curator.retry.RetryUntilElapsed;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.lang.Nullable;
import org.springframework.util.CollectionUtils;

import java.util.List;

/**
 * 分布式锁自动配置类
 */
@Configuration
@EnableConfigurationProperties(LockProperties.class)
@Import({LockInterceptorAutoConfiguration.class, CodecPropertiesCustomizer.class})
public class LockAutoConfiguration {

	private final List<PropertiesCustomizer> customizers;


	public LockAutoConfiguration(ObjectProvider<List<PropertiesCustomizer>> objectCustomizers) {
		this.customizers = objectCustomizers.getIfAvailable();
	}

	@Bean(LockCommonConstant.LOCK_CLIENT_BEAN_NAME)
	@ConditionalOnMissingBean(name = {LockCommonConstant.LOCK_CLIENT_BEAN_NAME})
	@ConditionalOnProperty(prefix = LockProperties.LOCK_PREFIX, name = LockProperties.PROPERTY_NAME, havingValue = LockScheme.LockSchemeConstant.REDIS)
	public RedissonClient redissonClient(PatternManager patternManager, LockProperties lockProperties) {
		// 对lockProperties进行自定义配置
		applyProperties(lockProperties);
		// 构造redisson配置
		InitConfig initConfig = new InitConfig(lockProperties.getPattern(), lockProperties, lockProperties.getCodec());
		return patternManager.initConfig(initConfig);
	}

	private void applyProperties(LockProperties lockProperties) {
		if (!CollectionUtils.isEmpty(this.customizers)) {
			for (PropertiesCustomizer customizer : this.customizers) {
				customizer.customize(lockProperties);
			}
		}
	}

	/* Lock Pattern configuration : 锁类型的初始化配置（单机，集群等） */

	@Bean
	public PatternManager patternManager(ObjectProvider<List<PatternProvider>> objectProviders) {
		return new RedisServerPatternManager(objectProviders.getIfAvailable());
	}

	/**
	 * redis单机模式配置
	 *
	 * @return
	 */
	@Bean
	public PatternProvider redisSingleProvider() {
		return new RedisSingleProvider();
	}

	/**
	 * redis集群模式配置
	 *
	 * @return
	 */
	@Bean
	public PatternProvider redisClusterProvider() {
		return new RedisClusterProvider();
	}

	/* zookeeper 客户端配置 */

	@Bean(LockCommonConstant.LOCK_CLIENT_BEAN_NAME)
	@ConditionalOnMissingBean(name = {LockCommonConstant.LOCK_CLIENT_BEAN_NAME})
	@ConditionalOnProperty(prefix = LockProperties.LOCK_PREFIX, name = LockProperties.PROPERTY_NAME, havingValue = LockScheme.LockSchemeConstant.ZOOKEEPER)
	public CuratorFramework zkClient(LockProperties lockProperties) {
		ZookeeperSchemeConfig zookeeperSchemeConfig = lockProperties.getZookeeper();
		CuratorFramework curatorClient = CuratorFrameworkFactory.newClient(zookeeperSchemeConfig.getAddress(),
				zookeeperSchemeConfig.getSessionTimeout(), zookeeperSchemeConfig.getConnectionTimeoutMs(),
				initZkRetryPolicy(zookeeperSchemeConfig));
		curatorClient.start();
		return curatorClient;
	}

	/**
	 * 初始化zk重试策略
	 *
	 * @param zookeeperSchemeConfig
	 * @return
	 */
	@SneakyThrows
	private RetryPolicy initZkRetryPolicy(ZookeeperSchemeConfig zookeeperSchemeConfig) {
		ZkRetryPolicy zkRetryPolicy = ZkRetryPolicyStore.getPolicy(zookeeperSchemeConfig.getRetryPolicy());
		if (zkRetryPolicy == ZkRetryPolicy.RETRY_EXPONENTIAL_BACKOFF) {
			ExponentialBackoffRetryConfig exponentialBackoffRetryConfig = zookeeperSchemeConfig
					.getExponentialBackoffRetry();
			return new ExponentialBackoffRetry(exponentialBackoffRetryConfig.getBaseSleepTimeMs(),
					exponentialBackoffRetryConfig.getMaxRetries(), exponentialBackoffRetryConfig.getMaxSleepMs());
		}
		if (zkRetryPolicy == ZkRetryPolicy.RETRY_FOREVER) {
			ForeverRetryConfig foreverRetryConfig = zookeeperSchemeConfig.getForeverRetry();
			return new RetryForever(foreverRetryConfig.getRetryIntervalMs());
		}
		if (zkRetryPolicy == ZkRetryPolicy.RETRY_NTIMES) {
			NTimesRetryConfig ntimesRetryConfig = zookeeperSchemeConfig.getNtimesRetry();
			return new RetryNTimes(ntimesRetryConfig.getN(), ntimesRetryConfig.getSleepMsBetweenRetries());
		}
		if (zkRetryPolicy == ZkRetryPolicy.RETRY_UNTIL_ELAPSED) {
			UntilElapsedRetryConfig untilElapsedRetryConfig = zookeeperSchemeConfig.getUntilElapsedRetry();
			return new RetryUntilElapsed(untilElapsedRetryConfig.getMaxElapsedTimeMs(),
					untilElapsedRetryConfig.getSleepMsBetweenRetries());
		}
		return null;
	}



	/* lock service configuration : 锁操作客户端的类型配置(zk/redis等  重入/红锁等) */

	@Bean
	@ConditionalOnClass(RedissonClient.class)
	public LockServiceManager lockServiceManager(ObjectProvider<List<LockServiceProvider>> lockServiceProviders, @Nullable RedissonClient redissonClient, @Nullable CuratorFramework zkClient) {
		return new LockProviderManager(lockServiceProviders.getIfAvailable(), redissonClient, zkClient);
	}

	/**
	 * redis可重入锁
	 *
	 * @return
	 */
	@Bean
	public LockServiceProvider redisReentrantLockServiceProvider() {
		return new RedisReentrantLockServiceProvider();
	}

	/**
	 * redis公平锁
	 *
	 * @return
	 */
	@Bean
	public LockServiceProvider redisFairLockServiceProvider() {
		return new RedisFairLockServiceProvider();
	}

	/**
	 * redis联锁
	 *
	 * @return
	 */
	@Bean
	public LockServiceProvider redisMultiLockServiceProvider() {
		return new RedisMultiLockServiceProvider();
	}

	/**
	 * zookeeper联锁
	 *
	 * @return
	 */
	@Bean
	public LockServiceProvider zookeeperMultiZkLockServiceProvider() {
		return new ZookeeperMultiZkLockServiceProvider();
	}

}
