package com.smart.hospital.auth.endpoint;

import cn.hutool.core.util.IdUtil;
import com.smart.hospital.common.core.constant.CacheConstants;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.JdkSerializationRedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.security.oauth2.common.exceptions.InvalidGrantException;
import org.springframework.security.oauth2.provider.OAuth2Authentication;
import org.springframework.security.oauth2.provider.code.AuthorizationCodeServices;


/**
 * 自定义生成授权码模式的code的实现类
 * 并通过redis存储
 */
@Slf4j
public class SmartRedisAuthorizationCodeService implements AuthorizationCodeServices {

	@Autowired
	private RedisTemplate redisTemplate;

	@Override
	public String createAuthorizationCode(OAuth2Authentication authentication) {
		// 1.首先生成code
		String uuidCode = IdUtil.fastUUID();
		log.info("生成的授权code为:{}", uuidCode);
		// 存储在redis中
		redisTemplate.setKeySerializer(new StringRedisSerializer());
		redisTemplate.setValueSerializer(new JdkSerializationRedisSerializer());
		redisTemplate.opsForValue().set(CacheConstants.OAUTH_CODE + uuidCode, authentication);
		return uuidCode;
	}

	@Override
	public OAuth2Authentication consumeAuthorizationCode(String code) throws InvalidGrantException {
		redisTemplate.setKeySerializer(new StringRedisSerializer());
		redisTemplate.setValueSerializer(new JdkSerializationRedisSerializer());
		OAuth2Authentication authentication = (OAuth2Authentication) redisTemplate.opsForValue().get(CacheConstants.OAUTH_CODE + code);
		log.info("通过code获取到的auth2Authentication为:{}", authentication);
		if (authentication == null) {
			throw new InvalidGrantException("Invalid authorization code: " + code);
		}
		return authentication;
	}
}
