package com.smart.hospital.common.pay.strategy.handler;

import cn.hutool.core.io.file.FileWriter;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import com.smart.hospital.common.pay.core.wx.WxPayApi;
import com.smart.hospital.common.pay.enums.PaymentMethod;
import com.smart.hospital.common.pay.model.WxHttpResponse;
import com.smart.hospital.common.pay.prop.WxPayV3Properties;
import com.smart.hospital.common.pay.source.PayAttribute;
import com.smart.hospital.common.pay.util.AesUtil;
import com.smart.hospital.common.pay.util.PayKit;
import com.smart.hospital.common.pay.util.WxPayKit;
import lombok.extern.slf4j.Slf4j;

import java.io.ByteArrayInputStream;
import java.nio.charset.StandardCharsets;
import java.security.cert.X509Certificate;

/**
 * 微信获取平台证书
 */
@Slf4j
public class WxCertV3StrategyHandler extends AbstractStrategyHandler {

	public WxCertV3StrategyHandler(WxPayV3Properties wxPayV3Properties) {
		super(wxPayV3Properties);
	}

	public WxCertV3StrategyHandler() {
	}

	@Override
	public Object execute(PayAttribute attribute, Object... params) throws Exception {
		log.info("微信获取平台证书...");
		WxHttpResponse response = WxPayApi.v3(attribute.getRequestMethod(),
				attribute.getPrefix().getType(),
				attribute.getSuffix().getType(),
				getWxPayV3Properties().getMchId(),
				getSerialNumber(),
				null,
				getWxPayV3Properties().getKeyPath(),
				"");
		if (response != null) {
			log.info("获取证书的基础信息为:{}", response);
			saveCert(response);
		}
		return response;
	}

	@Override
	public boolean apply(PayAttribute attribute) {
		return attribute.getPayment() == PaymentMethod.WX_CERT;
	}

	private void saveCert(WxHttpResponse response) throws Exception {
		String timestamp = response.getHeader("Wechatpay-Timestamp");
		String nonceStr = response.getHeader("Wechatpay-Nonce");
		String serialNumber = response.getHeader("Wechatpay-Serial");
		String signature = response.getHeader("Wechatpay-Signature");

		String body = response.getBody();
		int status = response.getStatus();

		log.info("serialNumber: {}", serialNumber);
		log.info("status: {}", status);
		log.info("body: {}", body);
		int isOk = 200;
		if (status == isOk) {
			JSONObject jsonObject = JSONUtil.parseObj(body);
			JSONArray dataArray = jsonObject.getJSONArray("data");
			// 默认认为只有一个平台证书
			JSONObject encryptObject = dataArray.getJSONObject(0);
			JSONObject encryptCertificate = encryptObject.getJSONObject("encrypt_certificate");
			String associatedData = encryptCertificate.getStr("associated_data");
			String cipherText = encryptCertificate.getStr("ciphertext");
			String nonce = encryptCertificate.getStr("nonce");
			String serialNo = encryptObject.getStr("serial_no");
			final String platSerialNo = savePlatformCert(associatedData, nonce, cipherText, getWxPayV3Properties().getPlatformCertPath());
			log.info("平台证书序列号: {} serialNo: {}", platSerialNo, serialNo);
		}
		// 根据证书序列号查询对应的证书来验证签名结果
		boolean verifySignature = WxPayKit.verifySignature(response, getWxPayV3Properties().getPlatformCertPath());
		System.out.println("verifySignature:" + verifySignature);
	}

	private String savePlatformCert(String associatedData, String nonce, String cipherText, String certPath) {
		try {
			AesUtil aesUtil = new AesUtil(getWxPayV3Properties().getApiKey3().getBytes(StandardCharsets.UTF_8));
			// 平台证书密文解密
			// encrypt_certificate 中的  associated_data nonce  ciphertext
			String publicKey = aesUtil.decryptToString(
					associatedData.getBytes(StandardCharsets.UTF_8),
					nonce.getBytes(StandardCharsets.UTF_8),
					cipherText
			);
			// 保存证书
			FileWriter writer = new FileWriter(certPath);
			writer.write(publicKey);
			// 获取平台证书序列号
			X509Certificate certificate = PayKit.getCertificate(new ByteArrayInputStream(publicKey.getBytes()));
			return certificate.getSerialNumber().toString(16).toUpperCase();
		} catch (Exception e) {
			e.printStackTrace();
			return e.getMessage();
		}
	}
}
