package com.smart.hospital.common.mq.config;

import com.smart.hospital.common.mq.handler.ConfirmCallbackHandler;
import com.smart.hospital.common.mq.handler.ReturnCallbackHandler;
import lombok.extern.slf4j.Slf4j;
import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.rabbit.core.RabbitTemplate;
import org.springframework.amqp.support.converter.Jackson2JsonMessageConverter;
import org.springframework.lang.Nullable;

/**
 * RabbitMQ发送端配置类
 */
@Slf4j
public class RabbitMqConfiguration {

	private final RabbitTemplate rabbitTemplate;

	private ConfirmCallbackHandler confirmCallbackHandler;

	private ReturnCallbackHandler returnCallbackHandler;

	public RabbitMqConfiguration(RabbitTemplate rabbitTemplate,
								 @Nullable ConfirmCallbackHandler confirmCallbackHandler,
								 @Nullable ReturnCallbackHandler returnCallbackHandler) {
		this.rabbitTemplate = rabbitTemplate;
		this.confirmCallbackHandler = confirmCallbackHandler;
		this.returnCallbackHandler = returnCallbackHandler;
	}

	public void init() {
		// 这里可以解决message.getMessageProperties().getCorrelationId()获取为null的问题
		rabbitTemplate.setCorrelationDataPostProcessor((message, correlationData) -> {
			if (correlationData != null && message != null) {
				MessageProperties messageProperties = message.getMessageProperties();
				String correlationDataId = correlationData.getId();
				messageProperties.setCorrelationId(correlationDataId);
			}
			return correlationData;
		});

		// 数据转换为json存入消息队列
		rabbitTemplate.setMessageConverter(new Jackson2JsonMessageConverter());

		// 消息成功发送到交换机Exchange通知,这个回调方法里面没有原始消息,相当于只是一个通知作用
		rabbitTemplate.setConfirmCallback((correlationData, ack, cause) -> {
			if (confirmCallbackHandler != null) {
				confirmCallbackHandler.confirmCallback(correlationData, ack, cause);
			}
		});

		// 确认消息从Exchange路由到Queue，注意:这是一个失败回调，只有消息从Exchange路由到Queue失败，才会回调这个方法
		rabbitTemplate.setMandatory(true);
		rabbitTemplate.setReturnCallback((message, replyCode, replyText, exchange, routingKey) -> {
			if (returnCallbackHandler != null) {
				returnCallbackHandler.returnCallback(message, replyCode, replyText, exchange, routingKey);
			}
		});
	}

	public void setConfirmCallbackHandler(ConfirmCallbackHandler confirmCallbackHandler) {
		this.confirmCallbackHandler = confirmCallbackHandler;
	}

	public void setReturnCallbackHandler(ReturnCallbackHandler returnCallbackHandler) {
		this.returnCallbackHandler = returnCallbackHandler;
	}
}
