package com.smart.hospital.common.im.configuration;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.smart.hospital.common.im.client.ImClient;
import com.smart.hospital.common.im.client.OkHttpImClient;
import lombok.extern.slf4j.Slf4j;
import okhttp3.ConnectionPool;
import okhttp3.OkHttpClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.util.Assert;

import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.util.concurrent.TimeUnit;

/**
 * IM配置类(请求客户端)
 */
@Slf4j
@EnableConfigurationProperties(value = {ImProperties.class})
public class ImAutoconfigure {

	/**
	 * 默认请求过期时间
	 */
	private static final int DEFAULT_TIMEOUT = 10;

	private static final int DEFAULT_POOL = 100;

	/**
	 * http请求客户端
	 */
	private OkHttpClient okHttpClient;

	@Autowired(required = false)
	public void setOkHttpClient(OkHttpClient okHttpClient) {
		this.okHttpClient = okHttpClient;
	}

	/**
	 * 设置控制器映射
	 *
	 * @param properties Im配置
	 * @return ImClient
	 */
	@Bean
	public ImClient imClient(ImProperties properties) {
		Long sdkAppId = properties.getSdkAppId();
		String key = properties.getKey();
		String identifier = properties.getIdentifier();

		Assert.notNull(sdkAppId, "sdkAppId cannot be null");
		Assert.notNull(key, "key cannot be null");
		Assert.notNull(identifier, "identifier cannot be null");

		return new OkHttpImClient(new ObjectMapper(), sdkAppId, key, identifier, client());
	}

	private OkHttpClient client() {
		if (okHttpClient != null) {
			return okHttpClient;
		}
		okHttpClient = createHttpClient();
		return okHttpClient;
	}

	/**
	 * 创建http请求客户端
	 *
	 * @return http请求客户端
	 */
	private static OkHttpClient createHttpClient() {
		OkHttpClient.Builder builder = new OkHttpClient.Builder();
		builder.connectTimeout(DEFAULT_TIMEOUT, TimeUnit.SECONDS);
		builder.connectionPool(pool());
		builder.readTimeout(DEFAULT_TIMEOUT, TimeUnit.SECONDS);
		builder.writeTimeout(DEFAULT_TIMEOUT, TimeUnit.SECONDS);
		SSLSocketFactory sslSocketFactory = createSslSocketFactory();
		if (sslSocketFactory != null) {
			builder.sslSocketFactory(sslSocketFactory, TrustAllManager.INSTANCE);
		}
		builder.hostnameVerifier((hostname, session) -> true);
		return builder.build();
	}

	private static SSLSocketFactory createSslSocketFactory() {
		try {
			SSLContext sc = SSLContext.getInstance("TLS");
			sc.init(null, new TrustManager[]{TrustAllManager.INSTANCE}, new SecureRandom());
			return sc.getSocketFactory();
		} catch (Exception e) {
			log.debug(e.getLocalizedMessage(), e);
			return null;
		}
	}

	public static ConnectionPool pool() {
		return new ConnectionPool(DEFAULT_POOL, 5, TimeUnit.MINUTES);
	}


	private static class TrustAllManager implements X509TrustManager {

		static final TrustAllManager INSTANCE = new TrustAllManager();

		@Override
		public void checkClientTrusted(X509Certificate[] chain, String authType) {
		}

		@Override
		public void checkServerTrusted(X509Certificate[] chain, String authType) {
		}

		@Override
		public X509Certificate[] getAcceptedIssuers() {
			return new X509Certificate[]{};
		}
	}
}
