package com.smart.hospital.common.im;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.ParameterizedType;
import java.util.HashMap;
import java.util.Map;

/**
 * IM请求模板抽象类
 *
 * @param <R>
 */
@Slf4j
public abstract class AbstractImRequest<R extends AbstractImResponse> {

	/**
	 * 解析响应数据
	 *
	 * @return
	 */
	public final R parseResponse(String responseData, ObjectMapper objectMapper) {
		R res;
		try {
			Class<R> responseClass = getResponseClass();
			res = objectMapper.readValue(responseData, responseClass);
		} catch (Exception e) {
			log.error("response parse error : {}", e.getLocalizedMessage());
			throw new ImException("response parse error", e);
		}
		if (res == null) {
			throw new ImException("response is empty");
		}
		parseResponseAfter(res);
		return res;
	}

	/**
	 * 获取响应class类型
	 * 解析数据用
	 *
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public Class<R> getResponseClass() {
		// 1.((ParameterizedType)type).getActualTypeArguments() 是用来的到类的模板参数的类型的？ 入T1, T2...返回表示此类型实际类型参数的 Type 对象的数组
		// 2.ParameterizedType是Type的子接口，表示一个有参数的类型，例如Collection<T>，Map<K,V>等。但实现上 ParameterizedType并不直接表示Collection<T>和Map<K,V>等，
		// 而是表示 Collection<String>和Map<String,String>等这种具体的类型。是不是看着眼熟，其实这就是我们常说的泛型。而ParameterizedType代表的是一个泛型的实例
		return (Class<R>) ((ParameterizedType) getClass().getGenericSuperclass()).getActualTypeArguments()[0];
	}

	/**
	 * 解析后置处理,用于处理复杂响应
	 *
	 * @param res
	 */
	protected void parseResponseAfter(R res) {
	}

	/**
	 * 接口请求地址
	 *
	 * @return
	 */
	public abstract String uri();

	/**
	 * 获取请求正文
	 *
	 * @param objectMapper
	 * @return
	 */
	public abstract byte[] requestBody(ObjectMapper objectMapper);

	/**
	 * 获取请求地址查询参数
	 *
	 * @return 请求地址查询参数
	 */
	public Map<String, String> query() {
		return new HashMap<>(8);
	}
}
