package org.liquibase.core.liquibase;

import liquibase.changelog.visitor.ChangeExecListener;
import liquibase.exception.LiquibaseException;
import lombok.extern.slf4j.Slf4j;
import org.liquibase.core.configuration.DynamicLiquibaseConfiguration;
import org.liquibase.core.configuration.DynamicLiquibaseProperties;
import org.liquibase.core.property.DynamicProperty;
import org.springframework.beans.MutablePropertyValues;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.config.ConstructorArgumentValues;
import org.springframework.beans.factory.support.AbstractBeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.GenericBeanDefinition;
import org.springframework.beans.factory.wiring.BeanWiringInfo;
import org.springframework.beans.factory.wiring.BeanWiringInfoResolver;
import org.springframework.beans.factory.wiring.ClassNameBeanWiringInfoResolver;
import org.springframework.boot.autoconfigure.liquibase.LiquibaseProperties;
import org.springframework.core.io.DefaultResourceLoader;

import javax.sql.DataSource;
import java.util.List;
import java.util.Objects;

@Slf4j
public class LiquibaseCreator implements DisposableBean {

    private final ConfigurableListableBeanFactory beanFactory;
    private final DynamicLiquibaseProperties properties;
    private final List<ChangeExecListener> listeners;
    private final List<DynamicProperty<?>> dynamicProperties;
    private final DataSource dataSource;

    public LiquibaseCreator(ConfigurableListableBeanFactory beanFactory,
                            DynamicLiquibaseProperties properties,
                            List<ChangeExecListener> listeners,
                            List<DynamicProperty<?>> dynamicProperties,
                            DataSource dataSource) {
        this.beanFactory = beanFactory;
        this.properties = properties;
        this.listeners = listeners;
        this.dynamicProperties = dynamicProperties;
        this.dataSource = dataSource;
    }

    public void createLiquibaseBeanDefinition(String applicationId) throws LiquibaseException {
        if (!properties.isEnabled()) {
            log.info("Init liquibase bean not enable.");
            return;
        }

        String beanName = getBeanName("default");

        if (beanFactory.containsBeanDefinition(beanName) || beanFactory.containsBean(beanName)) {
            log.info("Bean '{}' already exists, skipping creation.", beanName);
            DynamicSpringLiquibase liquibaseBean = (DynamicSpringLiquibase) beanFactory.getBean(beanName);
            liquibaseBean.afterPropertiesSet();
            return;
        }

        GenericBeanDefinition beanDefinition = new GenericBeanDefinition();
        beanDefinition.setBeanClass(DynamicSpringLiquibase.class);
        beanDefinition.setScope(BeanDefinition.SCOPE_SINGLETON);
        beanDefinition.setAutowireMode(AbstractBeanDefinition.AUTOWIRE_BY_NAME);

        // 设置构造函数参数
        ConstructorArgumentValues args = new ConstructorArgumentValues();
        args.addIndexedArgumentValue(0, properties.getConfig());
        args.addIndexedArgumentValue(1, applicationId);
        args.addIndexedArgumentValue(2, getTargetDataSource());
        beanDefinition.setConstructorArgumentValues(args);

        // 设置属性值（如果需要）
        MutablePropertyValues propertyValues = new MutablePropertyValues();
        // 如果需要设置其他属性，可以在这里添加
        propertyValues.add("listeners", listeners);
        propertyValues.add("dynamicProperties", dynamicProperties);
        beanDefinition.setPropertyValues(propertyValues);

        // 设置初始化方法（如果有）
        // beanDefinition.setInitMethodName("init");

        // 注册 BeanDefinition
        ((BeanDefinitionRegistry) beanFactory).registerBeanDefinition(beanName, beanDefinition);

        // 立即初始化这个Bean（对于单例Bean）
        if (!beanFactory.isCurrentlyInCreation(beanName)) {
            beanFactory.getBean(beanName);
        }

        log.info("Init liquibase bean definition success");
    }

    public void createLiquibase(String applicationId) {
        if (!properties.isEnabled()) {
            log.info("Init liquibase bean not enable.");
            return;
        }
        BeanWiringInfoResolver bwiResolver = new ClassNameBeanWiringInfoResolver();
        String beanName = getBeanName("default");
        DynamicSpringLiquibase springLiquibase = springLiquibase(properties.getConfig(), applicationId);
        springLiquibase.setBeanName(beanName);

        BeanWiringInfo bwi = bwiResolver.resolveWiringInfo(springLiquibase);
        if (bwi == null) {
            // Skip the bean if no wiring info given.
            log.info("Skip the bean if no wiring info given.");
            return;
        }
        if (bwi.indicatesAutowiring() || bwi.isDefaultBeanName() && !beanFactory.containsBean(beanName)) {
            beanFactory.autowireBeanProperties(springLiquibase, bwi.getAutowireMode(), bwi.getDependencyCheck());
            beanFactory.initializeBean(springLiquibase, beanName);
        } else {
            beanFactory.configureBean(springLiquibase, beanName);
        }

        if (!beanFactory.containsBean(beanName)) {
            beanFactory.registerSingleton(beanName, springLiquibase);
        }

        log.info("Init liquibase bean success");
    }

    private DynamicSpringLiquibase springLiquibase(LiquibaseProperties properties, String applicationId) {
        DynamicSpringLiquibase liquibase = new DynamicSpringLiquibase();
        liquibase.setDataSource(getTargetDataSource());
        liquibase.setChangeLog(properties.getChangeLog());
        liquibase.setClearCheckSums(properties.isClearChecksums());
        liquibase.setContexts(properties.getContexts());
        liquibase.setDefaultSchema(properties.getDefaultSchema());
        liquibase.setLiquibaseSchema(properties.getLiquibaseSchema());
        liquibase.setLiquibaseTablespace(properties.getLiquibaseTablespace());
        liquibase.setDatabaseChangeLogTable(properties.getDatabaseChangeLogTable());
        liquibase.setDatabaseChangeLogLockTable(properties.getDatabaseChangeLogLockTable());
        liquibase.setDropFirst(properties.isDropFirst());
        liquibase.setApplicationId(applicationId);
        liquibase.setShouldRun(properties.isEnabled());
        liquibase.setLabelFilter(properties.getLabels());
        liquibase.setChangeLogParameters(properties.getParameters());
        liquibase.setRollbackFile(properties.getRollbackFile());
        liquibase.setTestRollbackOnUpdate(properties.isTestRollbackOnUpdate());
        liquibase.setTag(properties.getTag());
        liquibase.setResourceLoader(new DefaultResourceLoader());
        liquibase.setListeners(listeners);
        liquibase.setDynamicProperties(dynamicProperties);
        return liquibase;
    }

    public DataSource getTargetDataSource() {
        Objects.requireNonNull(dataSource, "dataSource must not be null");
        return dataSource;
    }

    public String getBeanName(String liquibaseType) {
        return liquibaseType + DynamicLiquibaseConfiguration.SPRING_LIQUIBASE;
    }

    @Override
    public void destroy() throws Exception {
        String beanName = getBeanName("default");
        if (beanFactory.containsBean(beanName)) {
            DynamicSpringLiquibase liquibase = beanFactory.getBean(beanName, DynamicSpringLiquibase.class);
            liquibase.destroy();
        }
    }
}
