"""
@Project ：chatwoot-agent 
@File    ：__init__.py.py
@Author  ：Lijun
@Date    ：2025/8/19 10:09 
"""

from abc import ABC, abstractmethod

from cache.services import CacheService
from config.logger import get_logger
from handler.base_api import create_new_message, toggle_status, ragflow_agent
from model.chatwoot_model import MessageEvent

logger = get_logger(__name__)


# 事件数据
class Event:
    def __init__(self, event_type: str, data: MessageEvent):
        self.event_type = event_type
        self.data = data


# 消息数据
class Message:
    def __init__(self, message_type, message_status, data: MessageEvent):
        self.message_type = message_type
        self.message_status = message_status
        self.data = data


# 事件的处理器接口
class EventHandler(ABC):
    @abstractmethod
    def handle_event(self, event: Event):
        pass

    @abstractmethod
    def support(self, event: Event) -> bool:
        return False


# 消息的处理器接口
class MessageHandler(ABC):
    @abstractmethod
    def handle_message(self, message: Message):
        pass

    @abstractmethod
    def support(self, message: Message) -> bool:
        return False


# 消息已创建事件
class MessageCreatedEventHandler(EventHandler):

    def handle_event(self, event: Event):
        processor = MessageProcessor()
        message = Message(event.data.message_type, event.data.conversation.status, event.data)
        return processor.process(message=message)

    def support(self, event: Event) -> bool:
        return event.event_type == 'message_created'


# 对话已解决事件
class ConversationResolvedHandler(EventHandler):

    def handle_event(self, event: Event):
        logger.info(f"ConversationResolvedHandler处理事件消息.")
        if event.data is None:
            return
        contact_inbox = event.data.contact_inbox
        if contact_inbox is None or contact_inbox.id is None:
            return
        contact_id = contact_inbox.id
        CacheService.invalidate(f"c:cr:{contact_id}")

    def support(self, event: Event) -> bool:
        return event.event_type == 'conversation_resolved'


class NothingEventHandler(EventHandler):

    def handle_event(self, event: Event):
        logger.info(f"Event type:{event.event_type}, Nothing to do")

    def support(self, event: Event) -> bool:
        return True


class NormalMessageHandler(MessageHandler):

    def handle_message(self, message: Message):
        logger.info(f"NormalMessageHandler处理消息.")
        message_data = message.data
        if message_data is None:
            return
        message_sender = message_data.sender
        if message_sender is None:
            return
        # 发送给bot
        sender_id = message_sender.id
        agent_response = ragflow_agent(sender_id, message_data.content)
        # 发送给chatwoot
        conversation_id = message_data.conversation.id
        account_id = message_data.account.id
        # 发送给chatwoot
        create_new_message(account_id, conversation_id, agent_response)

    def support(self, message: Message) -> bool:
        if message.message_type == "incoming" and message.message_status == "pending":
            return True
        return False


class ToggleStatusHandler(MessageHandler):

    def handle_message(self, message: Message):
        logger.info(f"ToggleStatusHandler处理消息.")
        conversation_id = message.data.conversation.id
        account_id = message.data.account.id
        return toggle_status(account_id, conversation_id, "open")

    def support(self, message: Message) -> bool:
        if message.message_type == "incoming" and message.message_status == "pending":
            # 从redis获取需要转换状态的关键词
            content = message.data.content
            kw_words = CacheService.lrange("c:kw:toggle", 0, -1)
            logger.info(f"Redis中转换消息状态的关键词:{kw_words}")
            if any(key in content for key in kw_words):
                return True
        return False


class OtherMessageHandler(MessageHandler):

    def handle_message(self, message: Message):
        logger.info(f"Message type:{message.message_type}, status:{message.message_status},  Nothing to do")

    def support(self, message: Message) -> bool:
        return True


class MessageProcessor:

    def __init__(self):
        self._message_handlers = [
            ToggleStatusHandler(),
            NormalMessageHandler(),
            OtherMessageHandler()
        ]

    def process(self, message: Message):
        for handler in self._message_handlers:
            if handler.support(message):
                return handler.handle_message(message)
        raise ValueError(f"没有找到支持消息类型 {message.message_type} 和状态 {message.message_status} 的处理器")
