"""
@Project ：chatwoot-agent 
@File    ：logger.py
@Author  ：Lijun
@Date    ：2025/8/20 11:11 
"""
import logging
import time
from functools import wraps
from logging.handlers import RotatingFileHandler, TimedRotatingFileHandler
import os


class FlaskLogger:

    def __init__(self, app=None):
        self.app = app
        if app is not None:
            self.init_app(app)

    def init_app(self, app):
        """初始化应用日志"""
        config = app.config

        # 确保日志目录存在
        if not os.path.exists(config["LOG_DIR"]):
            os.makedirs(config["LOG_DIR"])

        # 配置根日志器
        root_logger = logging.getLogger()
        root_logger.setLevel(config["LOG_LEVEL"])

        # 清除现有的处理器
        for handler in root_logger.handlers[:]:
            root_logger.removeHandler(handler)

        # 创建格式化器
        formatter = logging.Formatter(config["LOG_FORMAT"])

        # 控制台处理器
        console_handler = logging.StreamHandler()
        console_handler.setLevel(config["LOG_LEVEL"])
        console_handler.setFormatter(formatter)
        root_logger.addHandler(console_handler)

        # 应用日志文件处理器（按大小分割）
        file_handler = RotatingFileHandler(
            config["APP_LOG_FILE"],
            maxBytes=10 * 1024 * 1024,  # 10MB
            backupCount=5,
            encoding='utf-8'
        )
        file_handler.setLevel(config["LOG_LEVEL"])
        file_handler.setFormatter(formatter)
        root_logger.addHandler(file_handler)

        # 错误日志文件处理器
        error_handler = RotatingFileHandler(
            config["ERROR_LOG_FILE"],
            maxBytes=10 * 1024 * 1024,
            backupCount=5,
            encoding='utf-8'
        )
        error_handler.setLevel(logging.ERROR)
        error_handler.setFormatter(formatter)
        root_logger.addHandler(error_handler)

        # 访问日志文件处理器（按天分割）
        access_handler = TimedRotatingFileHandler(
            config["ACCESS_LOG_FILE"],
            when='midnight',
            interval=1,
            backupCount=30,
            encoding='utf-8'
        )
        access_handler.setLevel(logging.INFO)
        access_handler.setFormatter(formatter)

        # 创建访问日志器
        access_logger = logging.getLogger('access')
        access_logger.setLevel(logging.INFO)
        access_logger.addHandler(access_handler)
        access_logger.propagate = False

        # 设置Flask应用日志器
        app.logger = root_logger

        # 禁用Werkzeug的默认日志
        logging.getLogger('werkzeug').setLevel(logging.WARNING)


def get_logger(name='app'):
    """获取日志器"""
    return logging.getLogger(name)


def log_execution_time(func):
    @wraps(func)
    def wrapper(*args, **kwargs):
        logger = get_logger("performance")
        start_time = time.time()

        try:
            result = func(*args, **kwargs)
            execution_time = time.time() - start_time
            if execution_time > 1.0:
                logger.warning(
                    f"Slow execution: {func.__name__} took {execution_time:.3f}s"
                )
            elif execution_time > 0.5:
                logger.info(
                    f"Execution time: {func.__name__} took {execution_time:.3f}s"
                )

            return result
        except Exception as e:
            execution_time = time.time() - start_time
            logger.error(
                f"Error in {func.__name__} after {execution_time:.3f}s: {str(e)}"
            )
            raise

    return wrapper


def setup_request_logging(app):
    """设置请求日志中间件"""
    access_logger = get_logger('access')

    @app.before_request
    def before_request():
        """记录请求开始"""
        from flask import request
        request.start_time = time.time()

    @app.after_request
    def after_request(response):
        """记录请求完成"""
        from flask import request

        # 计算处理时间
        process_time = time.time() - getattr(request, 'start_time', time.time())

        # 记录访问日志
        access_logger.info(
            f"{request.method} {request.path} - {response.status_code} "
            f"- {process_time:.3f}s - {request.remote_addr}"
        )

        return response
