"""
@Project ：chatwoot-agent 
@File    ：__init__.py.py
@Author  ：Lijun
@Date    ：2025/8/19 11:11 
"""
import sys
from importlib.util import spec_from_file_location, module_from_spec
from pathlib import Path
from typing import List

from flask import Blueprint, Flask, request
from flask_cors import CORS

from config.logger import FlaskLogger, setup_request_logging
from config.settings import config_dict
from cache import redis_cache

# 配置常量
API_VERSION = "v1"
PAGE_SUFFIX = "_app.py"


def register_page(page_path: Path, app: Flask) -> str:
    """
    动态注册单个蓝图页面
    """
    # 提取页面名称和模块名称
    page_name = page_path.stem.removesuffix("_app")
    module_name = ".".join(
        page_path.parts[page_path.parts.index("api"): -1] + (page_name,)
    )

    try:
        # 动态导入模块
        spec = spec_from_file_location(module_name, page_path)
        if spec is None or spec.loader is None:
            raise ImportError(f"无法为 {page_path} 创建模块规范")

        page_module = module_from_spec(spec)

        # 设置模块级变量
        setattr(page_module, "app", app)

        # 创建或获取蓝图
        blueprint = getattr(page_module, "manager", None)
        if blueprint is None:
            blueprint_name = getattr(page_module, "page_name", page_name)
            blueprint = Blueprint(blueprint_name, module_name)
            setattr(page_module, "manager", blueprint)

        sys.modules[module_name] = page_module
        spec.loader.exec_module(page_module)

        # 确定URL前缀
        url_prefix = determine_url_prefix(page_name)

        # 注册蓝图
        app.register_blueprint(blueprint, url_prefix=url_prefix)

        print(f"成功注册蓝图: {page_name} 在 {url_prefix}")
        return url_prefix

    except Exception as e:
        print(f"注册蓝图 {page_path} 失败: {str(e)}")
        raise


def determine_url_prefix(page_name: str) -> str:
    """
    根据文件路径确定URL前缀
    :param page_name: 页面名称
    :return: URL前缀
    """
    return f"/{API_VERSION}/{page_name}"


def search_pages_path(directory: Path) -> List[Path]:
    """
    搜索目录下的所有蓝图文件
    :param directory: 要搜索的目录
    :return: 找到的蓝图文件路径列表
    """
    if not directory.is_dir():
        return []

    return [
        path for path in directory.glob(f"*{PAGE_SUFFIX}")
        if path.is_file() and not path.name.startswith(".")
    ]


def get_pages_directories() -> List[Path]:
    """
    获取要搜索的目录列表
    :return: 目录路径列表
    """
    base_dir = Path(__file__).parent
    return [
        base_dir
    ]


def register_all_pages(app) -> List[str]:
    """
    注册所有蓝图页面
    :return: 所有注册的URL前缀列表
    """
    pages_dirs = get_pages_directories()
    return [
        register_page(page_path, app)
        for directory in pages_dirs
        for page_path in search_pages_path(directory)
    ]


def create_app(config_name="development"):
    """应用工厂函数"""
    app = Flask(__name__)
    # 全局跨域配置(接口层面: @cross_origin(origin='*'))
    # CORS(app, resources={r"/api/*": {"origins": "*"}})
    CORS(app, resources=r"/*")
    # 加载配置
    app.config.from_object(config_dict[config_name])
    # 初始化日志
    logger = FlaskLogger()
    logger.init_app(app)
    # 设置请求日志
    setup_request_logging(app)
    # 初始化缓存
    initialize_cache(app)
    # 注册蓝图
    register_blueprints(app)
    # 注册错误处理
    register_error_handlers(app)
    return app


def initialize_cache(app):
    redis_cache.init_app(app)


def register_blueprints(app):
    register_all_pages(app)


def register_error_handlers(app):
    """注册错误处理器"""
    from flask import jsonify
    import logging

    logger = logging.getLogger(__name__)

    @app.errorhandler(404)
    def not_found(error):
        logger.warning(f"404 Not Found: {request.path}")
        return jsonify({'error': 'Not found'}), 404

    @app.errorhandler(500)
    def internal_error(error):
        logger.error(f"500 Internal Error: {str(error)}")
        return jsonify({'error': 'Internal server error'}), 500

    @app.errorhandler(Exception)
    def handle_exception(error):
        logger.exception("Unhandled exception occurred")
        return jsonify({'error': 'Internal server error'}), 500
