import asyncio
import logging
import traceback
from typing import Any

from agent.tools.mcp.base import BaseMcpClient, create_mcp_client

logger = logging.getLogger(__name__)


class McpProxy:
    def __init__(self, config: dict[str, dict[str, Any]]):
        self.config = config
        self.clients: dict[str, BaseMcpClient] = {}
        self.tool_map: dict[str, str] = {}  # tool_name -> client_name
        self.tool_cache: list[dict[str, Any]] = []  # 缓存聚合后的工具列表


    async def initialize(self):
        for name, conf in self.config.items():
            client = create_mcp_client(name, conf)
            await client.initialize()
            logger.info(f"[{name}] 初始化成功")
            self.clients[name] = client
        await self._build_tool_cache()

    async def _build_tool_cache(self):
        """构建工具列表缓存并建立 tool -> client 映射"""
        self.tool_map.clear()
        self.tool_cache.clear()

        for name, client in self.clients.items():
            try:
                tools = await client.list_tools()
                for tool in tools:
                    tool_name = tool.name
                    self.tool_map[tool_name] = name
                    self.tool_cache.append({
                        "type": 'function',
                        "function": {
                            "name": tool.name,
                            "description": tool.description,
                            "parameters": tool.inputSchema
                        }
                    })
            except Exception as e:
                logger.warning(f"[{name}] 获取工具失败: {e}")

    async def list_tools(self) -> list[dict[str, Any]]:
        """返回缓存中的工具列表，每项为 {'client': name, 'tool': tool_obj}"""
        return self.tool_cache

    async def refresh_tools(self):
        """强制重新获取所有工具信息并更新缓存"""
        await self._build_tool_cache()
        logger.info("工具列表已刷新")

    async def call_tool(self, tool_name: str, arguments: dict[str, Any]) -> Any:
        """自动查找并调用工具"""
        try:
            client_name = self.tool_map.get(tool_name)
            if not client_name:
                raise ValueError(f"未找到工具: {tool_name}")
            client = self.clients[client_name]
            logger.info(f"调用 [{client_name}] 的工具: {tool_name}")
            res = await client.call_tool(tool_name, arguments)
        except Exception as e:
            traceback.print_exc()
            raise e
        return  res

    async def cleanup(self):
        for name, client in self.clients.items():
            await client.cleanup()
            logger.info(f"[{name}] 已清理资源")


CONFIG = {
    "FastGPT": {
        "url": "https://cloud.fastgpt.cn/api/mcp/app/oj7BgLJmYh45J17YxC7EEFAj/mcp"
    }
}
proxy = McpProxy(CONFIG)
if __name__ == "__main__":
    import anyio

    logging.basicConfig(level=logging.INFO)
    logging.getLogger("httpx").setLevel(logging.WARNING)


    async def main():

        await proxy.initialize()

        # 获取缓存中的所有工具
        tools = await proxy.list_tools()
        print(tools)
        # await proxy.initialize()
        # tools = await proxy.list_tools()
        # print(tools)

        # # 调用某个工具（会自动找）
        result = await proxy.call_tool("计算器", {"question": "1+1", "a": 1, "b": 1, "operator": "add"})
        print("调用结果:", result)

        # 刷新工具列表缓存
        # await proxy.refresh_tools()
        #
        # await proxy.cleanup()


    anyio.run(main)
